CREATE OR REPLACE PACKAGE BODY TBICDS.PCK_XFER IS
/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

/*transfer a patient admission to the central database*/
procedure XferPatientVisit(
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_nXferSystemID          in number,
      pi_nRegionID              in number,
      pi_nSiteID                in number,
      pi_vDFN                   in varchar2,
      
      pi_vVisitID               in number,
      pi_nVisitLocationID       in number,
      pi_vVisitLocationName     in varchar2,
      pi_vVisitTimeStamp        in varchar2,
      pi_vVisitStatus           in varchar2,           
      pi_vVisitType             in varchar2,               
      
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2)
  is

    v_vSQLInsert constant varchar2(4000) :=
       'insert into
        patient_visit( REGION_ID,
                       SITE_ID,
                       DFN,
                          
                       VISIT_ID,
                       VISIT_LOCATION_ID,
                       VISIT_LOCATION_NAME,
                       VISIT_TIMESTAMP,
                       VISIT_STATUS,
                       VISIT_TYPE,
                       VISIT_DATETIME
                          )
          values(:REGION_ID,
                 :SITE_ID,
                 :DFN,
                 
                 :VISIT_ID,
                 :VISIT_LOCATION_ID,
                 :VISIT_LOCATION_NAME,
                 :VISIT_TIMESTAMP,
                 :VISIT_STATUS,
                 :VISIT_TYPE,
                 :VISIT_DATETIME)';

     --todo:
     v_vSQLUpdate constant varchar2(4000) := '';
     
     v_vSQLCount varchar2(4000) := '';
     v_nCount number := 0;

  begin

     v_nCount := 0;
     po_nStatusCode    := 0;
     po_vStatusComment := '';
     
     --check that a record exisits
     select count(*) into v_nCount 
     from patient_visit 
    WHERE region_id =  pi_nRegionID
     and site_id = pi_nSiteID
     and DFN = pi_vDFN
     and VISIT_LOCATION_ID = pi_nVisitLocationID
     and VISIT_TIMESTAMP = pi_vVisitTimeStamp;
          
     --insert if the record is not there
     if (v_nCount < 1) then

        --execute the sql with bind vars
        EXECUTE IMMEDIATE v_vSQLInsert
        USING pi_nRegionID,
              pi_nSiteID,
              pi_vDFN,
              
              pi_vVisitID,
              pi_nVisitLocationID,
              pi_vVisitLocationName,
              pi_vVisitTimeStamp,
              pi_vVisitStatus,           
              pi_vVisitType,
              to_date(pi_vVisitTimeStamp, 'yyyymmdd.hh24miss');   
                 
        commit;

      else

        --no update just insert
        commit;

      end if;

exception
    when others then
      po_nStatusCode    := PCK_COMMON.c_nStatus_Error;
      po_vStatusComment := '';

end;


/*transfer a patient admission to the central database*/
procedure XferPatientAdmission(
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_nXferSystemID          in number,
      pi_nRegionID              in number,
      pi_nSiteID                in number,
      pi_vDFN                   in varchar2,
      
      pi_nAdmitLocationID       in number,
      pi_vAdmitLocationName     in varchar2,
      pi_vAdmitType             in varchar2,
      pi_vAdmitTimeStamp        in varchar2,
      pi_vDischargeTimeStamp   in varchar2,                          
      
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2)
  is

    v_vSQLInsert constant varchar2(4000) :=
       'insert into
        patient_admission( REGION_ID,
                           SITE_ID,
                           DFN,
                          
                           ADMIT_LOCATION_ID,
                           ADMIT_LOCATION_NAME,
                           ADMIT_TYPE,
                           ADMIT_TIMESTAMP,
                           ADMIT_DISCHARGE_TIMESTAMP,
                           ADMIT_DATETIME
                          )
          values(:REGION_ID,
                 :SITE_ID,
                 :DFN,
                 
                 :ADMIT_LOCATION_ID,
                 :ADMIT_LOCATION_NAME,
                 :ADMIT_TYPE,
                 :ADMIT_TIMESTAMP,
                 :ADMIT_DISCHARGE_TIMESTAMP,
                 :ADMIT_DATETIME)';

     --todo:
     v_vSQLUpdate constant varchar2(4000) := '';
    

     v_vSQLCount varchar2(4000) := '';
     v_nCount number := 0;

  begin

     v_nCount := 0;
     po_nStatusCode    := 0;
     po_vStatusComment := '';
     
     --check that a record exisits
     select count(*) into v_nCount 
     from patient_admission 
     WHERE region_id = pi_nRegionID 
      and site_id = pi_nSiteID 
      and DFN = pi_vDFN 
      and ADMIT_LOCATION_ID = pi_nAdmitLocationID 
      and ADMIT_TIMESTAMP = pi_vAdmitTimeStamp; 
     
     --insert if the record is not there
     if (v_nCount < 1) then

        --execute the sql with bind vars
        EXECUTE IMMEDIATE v_vSQLInsert
        USING pi_nRegionID,
              pi_nSiteID,
              pi_vDFN,
              
              pi_nAdmitLocationID,
              pi_vAdmitLocationName,
              pi_vAdmitType,
              pi_vAdmitTimeStamp,
              pi_vDischargeTimeStamp,
              to_date(pi_vAdmitTimeStamp, 'yyyymmdd.hh24miss');  
                 
        commit;

      else

        --execute the sql with bind vars
        --todo:
        --EXECUTE IMMEDIATE v_vSQLUpdate
        --USING pi_vRegionName,
        --      pi_nXferSystemID,
        --      pi_nRegionID;
        commit;

      end if;

exception
    when others then
      po_nStatusCode    := PCK_COMMON.c_nStatus_Error;
      po_vStatusComment := '';

end;

/*transfers a patient consult to the central database*/
procedure XferPatientConsult(
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_nXferSystemID          in number,
      pi_nRegionID              in number,
      pi_nSiteID                in number,
      pi_vDFN                   in varchar2,
      
      pi_vConsultID             in varchar2,
      pi_vConsultTimeStamp      in varchar2,
      pi_vConsultStatus         in varchar2,
      pi_vConsultTitle          in varchar2,
      pi_vConsultText           in varchar2,                          
      
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2)
  is

    v_vSQLInsert constant varchar2(4000) :=
         'insert into
          patient_consult( REGION_ID,
                           SITE_ID,
                           DFN,
                          
                           CONSULT_ID,
                           CONSULT_TIMESTAMP,
                           CONSULT_STATUS,
                           CONSULT_TITLE,
                           CONSULT_TEXT,
                           CONSULT_DATETIME
                          )
          values(:REGION_ID,
                 :SITE_ID,
                 :DFN,
                 
                 :CONSULT_ID,
                 :CONSULT_TIMESTAMP,
                 :CONSULT_STATUS,
                 :CONSULT_TITLE,
                 :CONSULT_TEXT,
                 :CONSULT_DATETIME)';

     --todo:
     v_vSQLUpdate constant varchar2(4000) := '';
     
     v_vSQLCount varchar2(4000) := '';
     v_nCount number := 0;

  begin

     v_nCount := 0;
     po_nStatusCode    := 0;
     po_vStatusComment := '';
     
     --check that a record exisits
     select count(*) into v_nCount  
     from patient_consult 
     WHERE region_id = pi_nRegionID 
      and site_id = pi_nSiteID 
      and DFN = pi_vDFN 
      and consult_id = pi_vConsultID;
     
     --insert if the record is not there
     if (v_nCount < 1) then

        --execute the sql with bind vars
        EXECUTE IMMEDIATE v_vSQLInsert
        USING pi_nRegionID,
              pi_nSiteID,
              pi_vDFN,
              
              pi_vConsultID,
              pi_vConsultTimeStamp,
              pi_vConsultStatus,
              pi_vConsultTitle,
              pi_vConsultText,
              to_date(pi_vConsultTimeStamp, 'yyyymmdd.hh24miss');                 
        commit;

      else

        --execute the sql with bind vars
        --todo:
        --EXECUTE IMMEDIATE v_vSQLUpdate
        --USING pi_vRegionName,
        --      pi_nXferSystemID,
        --      pi_nRegionID;
        commit;

      end if;

exception
    when others then
      po_nStatusCode    := PCK_COMMON.c_nStatus_Error;
      po_vStatusComment := '';

end;

/*transfers a patient appointment to the database*/
procedure XferPatientAppointment(
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_nXferSystemID          in number,
      pi_nRegionID              in number,
      pi_nSiteID                in number,
      pi_vAppointmentID         in varchar2,
      pi_vDFN                   in varchar2,
      pi_nApptClinicID          in number,
      pi_nApptSiteID            in number,
      pi_vApptCurrentStatus     in varchar2,
      pi_vApptPurpose           in varchar2,
      pi_vApptStatus            in varchar2,
      pi_vApptText              in varchar2,
      pi_vApptTimeStamp         in varchar2,
      pi_vApptTitle             in varchar2,
      pi_vApptType              in varchar2,
      pi_vApptVisitID           in varchar2,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2)
  is

    v_vSQLInsert constant varchar2(4000) :=
         'insert into
          patient_appointment( REGION_ID,
                               SITE_ID,
                               APPT_ID,
                               DFN,
                               APPT_CLINIC_ID,
                               APPT_SITE_ID,
                               APPT_CURRENT_STATUS,
                               APPT_PURPOSE,
                               APPT_STATUS,
                               APPT_TEXT,
                               APPT_TIMESTAMP,
                               APPT_TITLE,
                               APPT_TYPE,
                               APPT_VISIT_ID,
                               APPT_DATETIME
                             )
          values(:REGION_ID,
                 :SITE_ID,
                 :APPT_ID,
                 :DFN,
                 :APPT_CLINIC_ID,
                 :APPT_SITE_ID,
                 :APPT_CURRENT_STATUS,
                 :APPT_PURPOSE,
                 :APPT_STATUS,
                 :APPT_TEXT,
                 :APPT_TIMESTAMP,
                 :APPT_TITLE,
                 :APPT_TYPE,
                 :APPT_VISIT_ID,
                 :APPT_DATETIME)';

     --todo:
     v_vSQLUpdate constant varchar2(4000) := '';
     
     v_vSQLCount varchar2(4000) := '';
     v_nCount number := 0;

  begin

     v_nCount := 0;
     po_nStatusCode    := 0;
     po_vStatusComment := '';
     
     --check that a record exisits
     select count(*) into v_nCount  
     from patient_appointment 
     WHERE region_id = pi_nRegionID 
      and site_id = pi_nSiteID 
      and DFN = pi_vDFN 
      and appt_id = pi_vAppointmentID;     
  
     --insert if the record is not there
     if (v_nCount < 1) then

        --execute the sql with bind vars
        EXECUTE IMMEDIATE v_vSQLInsert
        USING pi_nRegionID,
              pi_nSiteID,
              pi_vAppointmentID,
              pi_vDFN,
              pi_nApptClinicID,
              pi_nApptSiteID,
              pi_vApptCurrentStatus,
              pi_vApptPurpose,
              pi_vApptStatus,
              pi_vApptText,
              pi_vApptTimeStamp,
              pi_vApptTitle,
              pi_vApptType,
              pi_vApptVisitID,
              to_date(pi_vApptTimeStamp, 'yyyymmdd.hh24miss');
                 
        commit;

      else

        --execute the sql with bind vars
        --todo:
        --EXECUTE IMMEDIATE v_vSQLUpdate
        --USING pi_vRegionName,
        --      pi_nXferSystemID,
        --      pi_nRegionID;
        commit;

      end if;

exception
    when others then
      po_nStatusCode    := PCK_COMMON.c_nStatus_Error;
      po_vStatusComment := '';

end;

/*
transfers a clinic to the database
*/
procedure XferClinic(
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_nXferSystemID          in number,
      pi_nRegionID              in number,
      pi_nSiteID                in number,
      pi_nClinicID              in number,
      pi_vClinicLabel           in varchar2,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2)
  is

    v_vSQLInsert constant varchar2(4000) :=
         'insert into
          tbicds.utl_Clinic(Clinic_id,
                               Clinic_label,
                               xfer_system_id,
                               xfer_date,
                               region_id,
                               site_id
                          )
          values(:Clinic_ID,
                 :Clinic_LABEL,
                 :XFER_SYSTEM_ID,
                 sysdate,
                 :REGION_ID,
                 :SITE_ID)';

     v_vSQLUpdate constant varchar2(4000) :=
         'update tbicds.utl_Clinic
          set Clinic_label = :Clinic_LABEL,
              xfer_system_id = :XFER_SYSTEM_ID,
              xfer_date = sysdate
          where Clinic_id = :Clinic_ID
           and region_id = :REGION_ID
           and site_id = :SITE_ID';

     v_vSQLCount varchar2(4000) := '';
     v_nCount number := 0;

  begin

     --error messages
     v_nCount := 0;
     po_nStatusCode    := 0;
     po_vStatusComment := '';
     
     --check that a record exisits
     select count(*) into v_nCount 
     from tbicds.utl_Clinic 
     WHERE clinic_id = pi_nClinicID 
      and region_id = pi_nRegionID 
      and site_id = pi_nSiteID;

     --insert if the record is not there
     if (v_nCount < 1) then

        --execute the sql with bind vars
        EXECUTE IMMEDIATE v_vSQLInsert
        USING pi_nClinicID,
              pi_vClinicLabel,
              pi_nXferSystemID,
              pi_nRegionID,
              pi_nSiteID;
        commit;

      else

        --execute the sql with bind vars
        EXECUTE IMMEDIATE v_vSQLUpdate
        USING pi_vClinicLabel,
              pi_nXferSystemID,
              pi_nClinicID,
              pi_nRegionID,
              pi_nSiteID;

        commit;

      end if;

exception
    when others then
      po_nStatusCode    := PCK_COMMON.c_nStatus_Error;
      po_vStatusComment := 'Error transferring clinic';
      
end;

/*Transfers a note title from an external source such as MDWS to the database
*/
procedure XferNoteTitle(
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_nXferSystemID          in number,
      pi_nRegionID              in number,
      pi_nSiteID                in number,
      pi_nNoteTitleTag          in number,
      pi_vNoteTitleLabel        in varchar2,
      pi_nIsConsult             in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2)
  is

    v_vSQLInsert constant varchar2(4000) :=
         'insert into
          tbicds.utl_note_title(note_title_tag,
                               note_title_label,
                               xfer_system_id,
                               xfer_date,
                               region_id,
                               site_id,
                               is_consult
                          )
          values(:note_title_tag,
                 :NOTE_TITLE_LABEL,
                 :XFER_SYSTEM_ID,
                 sysdate,
                 :REGION_ID,
                 :SITE_ID,
                 :IS_CONSULT)';

     v_vSQLUpdate constant varchar2(4000) :=
         'update tbicds.utl_note_title
          set note_title_tag = :NOTE_TITLE_TAG,
              xfer_system_id = :XFER_SYSTEM_ID,
              xfer_date = sysdate,
              is_consult = :IS_CONSULT
          where note_title_label = :NOTE_TITLE_LABEL
          and  region_id = :REGION_ID
          and  site_id = :SITE_ID';

     v_vSQLCount varchar2(4000) := '';
     v_nCount number := 0;

  begin

     v_nCount := 0;
     po_nStatusCode    := 0;
     po_vStatusComment := '';
     
     --check that a record exisits
     select count(*) into v_nCount 
     from tbicds.utl_note_title
     WHERE note_title_label = pi_vNoteTitleLabel
      and site_id = pi_nSiteID
      and region_id = pi_nRegionID;
     
     --insert if the record is not there, otherwise update
     if (v_nCount < 1) then

        --execute the sql with bind vars
        EXECUTE IMMEDIATE v_vSQLInsert
        USING pi_nNoteTitleTag,
              pi_vNoteTitleLabel,
              pi_nXferSystemID,
              pi_nRegionID,
              pi_nSiteID,
              pi_nIsConsult;
        commit;

      else

        --execute the sql with bind vars
        EXECUTE IMMEDIATE v_vSQLUpdate
        USING pi_nNoteTitleTag,
              pi_nXferSystemID,
              pi_nIsConsult,
              pi_vNoteTitleLabel,
              pi_nRegionID,
              pi_nSiteID;
        commit;

      end if;

exception
    when others then
      po_nStatusCode    := PCK_COMMON.c_nStatus_Error;
      po_vStatusComment := 'Error transferring note title';
     
end;

/*
Transfers a region from an external source such as MDWS to the database
*/
procedure XferRegion(
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_nXferSystemID          in number,
      pi_nRegionID              in number,
      pi_vRegionName            in varchar2,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2)
  is

    v_vSQLInsert constant varchar2(4000) :=
         'insert into
          utl_region( region_id,
                             region_name,
                             xfer_system_id,
                             xfer_date
                          )
          values(:region_id,
                 :region_name,
                 :xfer_system_id,
                 sysdate)';

     v_vSQLUpdate constant varchar2(4000) :=
         'update utl_region
          set region_name = :region_name,
              xfer_system_id = :XFER_SYSTEM_ID,
              xfer_date = sysdate
          where region_id = :region_id';

     v_vSQLCount varchar2(4000) := '';
     v_nCount number := 0;

  begin

     v_nCount := 0;
     po_nStatusCode    := 0;
     po_vStatusComment := '';
     
     --check that a record exisits
     select count(*) into v_nCount 
     from utl_region  
     WHERE region_id = pi_nRegionID;
     
     --insert if the record is not there
     if (v_nCount < 1) then

        --execute the sql with bind vars
        EXECUTE IMMEDIATE v_vSQLInsert
        USING pi_nRegionID,
              pi_vRegionName,
              pi_nXferSystemID;
        commit;

      else

        --execute the sql with bind vars
        EXECUTE IMMEDIATE v_vSQLUpdate
        USING pi_vRegionName,
              pi_nXferSystemID,
              pi_nRegionID;
        commit;

      end if;

exception
    when others then
      po_nStatusCode    := PCK_COMMON.c_nStatus_Error;
      po_vStatusComment := '';

end;

/*
Transfers a site from an external source such as MDWS to the database
*/
procedure XferSite(
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_nXferSystemID          in number,
      pi_nRegionID              in number,
      pi_nSiteID                in number,
      pi_vSiteName            in varchar2,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2)
  is

    v_vSQLInsert constant varchar2(4000) :=
         'insert into
          utl_site( region_id,
                           site_id,
                           site_name,
                           xfer_system_id,
                           xfer_date
                          )
          values(:region_id,
                 :site_id,
                 :site_name,
                 :xfer_system_id,
                 sysdate)';

     v_vSQLUpdate constant varchar2(4000) :=
         'update utl_site
          set site_name = :site_name,
              xfer_system_id = :XFER_SYSTEM_ID,
              xfer_date = sysdate
          where region_id = :region_id
            and site_id = :site_id';

     v_vSQLCount varchar2(4000) := '';
     v_nCount number := 0;

  begin

     v_nCount := 0;
     po_nStatusCode    := 0;
     po_vStatusComment := '';
     
     --check that a record exisits
     select count(*) into v_nCount 
     from utl_site 
     WHERE region_id = pi_nRegionID 
      AND site_id = pi_nSiteID;

     --insert if the record is not there
     if (v_nCount < 1) then

        --execute the sql with bind vars
        EXECUTE IMMEDIATE v_vSQLInsert
        USING pi_nRegionID,
              pi_nSiteID,
              pi_vSiteName,
              pi_nXferSystemID;
        commit;

      else

        --execute the sql with bind vars
        EXECUTE IMMEDIATE v_vSQLUpdate
        USING pi_vSiteName,
              pi_nXferSystemID,
              pi_nRegionID,
              pi_nSiteID;
        commit;

      end if;

exception
    when others then
      po_nStatusCode    := PCK_COMMON.c_nStatus_Error;
      po_vStatusComment := '';
 
end;


/*
Transfers a user from an external source such as MDWS to the database
*/
procedure XferUser(
      pi_vSessionID           in varchar2,
      pi_vSessionClientIP     in varchar2,
      pi_nUserID              in number,
      pi_nXferSystemID        in number,
      pi_vDUZ                 in varchar2,
      pi_vLastName            in varchar2,
      pi_vFirstName           in varchar2,
      pi_vFirstLastName       in varchar2, 
      pi_vGreeting            in varchar2,
      pi_vTitle               in varchar2,
      pi_nRegionID            in number,
      pi_nSiteID              in number,
      pi_vUserClass           in varchar2,
      po_nUserID              out number,
      po_nStatusCode          out number,
      po_vStatusComment       out varchar2)
 is

    v_vSQLFXUserInsert constant varchar2(4000) :=
         'insert into
          fx_user( fx_user_id,
                   duz,
                   region_id,
                   site_id,
                   greeting,
                   xfer_system_id,
                   xfer_date)
          values(:fx_user_id,
                 :duz,
                 :region_id,
                 :site_id,
                 :greeting,
                 :xfer_system_id,
                 sysdate)';
      
      --insert statement for app_user table
      --todo: work out DMIS
      v_vSQLSuatUserInsert constant varchar2(4000) :=
         'insert into
          app_user( email,
                     uidpwd,
                     rank,
                     name,
                     title,
                     provider_id,
                     military_service_id,
                     corps,
                     unit,
                     squadron,
                     phone,
                     dims_id,
                     entry_date,
                     locked,
                     must_change_password,
                     fx_user_id,
                     supervisor_id,
                     graph_pref,
                     xfer_system_id,
                     xfer_date)
          values('''',
                 ''N/A'',
                 ''N/A'',
                 :name,
                 :title,
                 :provider_id,
                 0,
                 ''N/A'',
                 ''N/A'',
                 ''N/A'',
                 '''',
                 ''0000'', 
                 sysdate,
                 0,
                 0,
                 :fx_user_id,
                 ''N/A'',
                 0,
                 :xfer_system_id,
                 sysdate)';

     --fx_user update
     v_vSQLFXUserUpdate constant varchar2(4000) :=
         'update fx_user
          set duz = :duz,
              region_id = :region_id,
              site_id = :site_id,
              greeting = :greeting,
              xfer_system_id = :xfer_system_id,
              xfer_date =  sysdate
          where fx_user_id = :fx_user_id';

     --app_user update
     v_vSQLSuatUserUpdate constant varchar2(4000) :=
         'update app_user
          set name = :name,
              title = :title,
              xfer_system_id = :XFER_SYSTEM_ID,
              xfer_date = sysdate
          where fx_user_id = :fx_user_id';
          
     v_nCount number := 0;
     v_nFXUserID number := 0;
    
  begin

     po_nStatusCode    := 0;
     po_vStatusComment := '';
     
     --check that a record exisits
     select count(*) into v_nCount
     from fx_user t
     where t.region_id = pi_nRegionID
     and t.site_id = pi_nSiteID
     and t.duz = pi_vDUZ;
     
     --insert if the record is not there
     if (v_nCount < 1) then
        
        --get a new fx_user_id for this user
        select seqfxuserid.nextval into v_nFXUserID from dual;

        --execute fx_user insert sql with bind vars
        EXECUTE IMMEDIATE v_vSQLFXUserInsert
        USING v_nFXUserID,
              pi_vDUZ,
              pi_nRegionID,
              pi_nSiteID,
              pi_vGreeting,
              pi_nXferSystemID;
        
        --execute the app_user insert sql with bind vars
        EXECUTE IMMEDIATE v_vSQLSuatUserInsert
        USING pi_vFirstLastName,
              pi_vTitle,
              PCK_SYSTEM_ID.GetNewProviderID(pi_vFirstName,
                                             pi_vLastName),
              v_nFXUserID,
              pi_nXferSystemID;
                      
        --load user rights, todo: research this some may 
        --want to pass the rights in?
        insert into fx_user_rights(fx_user_id,
                                   user_rights,
                                   read_only,
                                   user_type)
        values( v_nFXUserID,             
                65204351,
                0,
                1);
        
        commit;
        --keep the user id
        po_nUserID := v_nFXUserID;

      else
        
        --get the fx_user_id for the update
        select fx_user_id into v_nFXUserID
        from fx_user t
        where t.region_id = pi_nRegionID
        and t.site_id = pi_nSiteID
        and t.duz = pi_vDUZ;

        --execute the fx_user sql with bind vars 
        EXECUTE IMMEDIATE v_vSQLFXUserUpdate
        USING pi_vDUZ,
              pi_nRegionID,
              pi_nSiteID,
              pi_vGreeting,
              pi_nXferSystemID,
              v_nFXUserID;
        
        --execute the app_user sql with bind vars        
        EXECUTE IMMEDIATE v_vSQLSuatUserUpdate
        USING pi_vFirstLastName,
              pi_vTitle,
              pi_nXferSystemID,
              v_nFXUserID;
        
        commit;
     
        --keep the user id
        po_nUserID := v_nFXUserID;
        
      end if;

exception
    when others then
      rollback;
      po_nStatusCode    := PCK_COMMON.c_nStatus_Error;
      po_vStatusComment := '';
 
end;


/*Transfers a patient from an external source such as MDWS to the database
*/
procedure XferPatient (   pi_vSessionID        in varchar2,
                          pi_vSessionClientIP  in varchar2,
                          pi_nUserID           in number,
                          pi_nXferSystemID     in number,
                          pi_vKey              in varchar2,
                          pi_nRegionID         in number,
                          pi_nSiteID           in number,
                          pi_vMDWSPatientID              in varchar2,
                          pi_vSSN              in varchar2,
                          pi_dtDOB             in date,
                          pi_vFirstName        in varchar2,
                          pi_vFullName         in varchar2,
                          pi_vLastName         in varchar2,
                          pi_vMI               in varchar2,
                          pi_nSex              in number,
                          pi_vHomeAddr1        in varchar2,
                          pi_vHomeAddr2        in varchar2,
                          pi_vHomeCity         in varchar2,
                          pi_vHomeState        in varchar2,
                          pi_vHomeZip          in varchar2,
                          pi_vHomePhone        in varchar2,
                          po_vPatientID        out varchar2,
                          po_nStatusCode       out number,
                          po_vStatusComment    out varchar2
                       )
is

   v_nRelationshipID number := 0;
   v_nCount number := 0;
   v_vSQLCount varchar2(2000) := '';
   v_vSex varchar2(20) := '';
   v_vProviderID varchar2(255) := '';
   
   --get the patient id and encounter id from an sp call
   v_vPatientID varchar2(500) := '';
   v_vEncounterID varchar2(500) := '';
   
begin
  
    po_vPatientID := '';
    po_nStatusCode    := 0;
    po_vStatusComment := '';
    v_nCount := 0;
    v_vSQLCount := '';
    v_vSex := 'U';
    v_vProviderID := '';
    v_nRelationshipID := 0;

    --get count
    SELECT count(*) into v_nCount 
      FROM patient_demographics 
     WHERE DFN = pi_vMDWSPatientID
       and region_id = pi_nRegionID
       and site_id = pi_nSiteID;

    --translate gender
    if(pi_nSex = 1) then
      v_vSex := 'M';
    end if;
    if(pi_nSex = 2) then
      v_vSex := 'F';
    end if;

    --default the provider to the person logged in
    select t.provider_id into v_vProviderID
    from app_user t
    where t.fx_user_id = pi_nuserid;

    if(v_nCount < 1) then
    
        --get a new patient id by calling the PCK_SYSTEM_ID package 
        v_vPatientID := PCK_SYSTEM_ID.GetNewPatientID(pi_vFirstName,
                                                      pi_vLastName,
                                                      to_char(pi_dtdob,'mm/dd/yyyy'),
                                                      v_vSex,
                                                      pi_vSSN);
                                                      
                                                      

        --insert new record into the patient demographics table
        insert into patient_demographics( patient_id,
                                          dfn,
                                          region_id,
                                          site_id,
                                          first_name,
                                          last_name,
                                          mi,
                                          ssn,
                                          gender,
                                          dob,
                                          provider_id)
        values
        (
           upper(v_vPatientID),
           pi_vMDWSPatientID,
           pi_nRegionID,
           pi_nSiteID,
           fnc_utl_encstr(upper(pi_vFirstName), pi_vKey, upper(v_vPatientID)),
           fnc_utl_encstr(upper(pi_vLastName), pi_vKey, upper(v_vPatientID)),
           fnc_utl_encstr(upper(pi_vMI), pi_vKey, upper(v_vPatientID)),
           fnc_utl_encstr(upper(pi_vSSN), pi_vKey, upper(v_vPatientID)),
           fnc_utl_encstr(v_vSex, pi_vKey, upper(v_vPatientID)),
           fnc_utl_encstr(to_char(pi_dtdob,'mm/dd/yyyy'), pi_vKey, upper(v_vPatientID)),
           v_vProviderID
        );
        commit;
        
        --pass the patient id back to the caller
        po_vPatientID := v_vPatientID;                        

        --Create Initial Patient Sponsor Record Needed By Assessments --Assessments can only to Updates
        /*insert into patient_sponsor
          (patient_id, relationship_id, last_updated, last_updated_by)
        values
          (v_vPatientID, v_nRelationshipID, sysdate, pi_nUserID);
        commit;*/

        --Create Initial Emergency Contact Record Needed By Assessments --Assessments can only to Updates
        insert into patient_emergency_contact
          (patient_id, relationship_id, last_updated, last_updated_by)
        values
          (v_vPatientID, v_nRelationshipID, sysdate, pi_nUserID);
        commit;

        --insert a new treatment record
 --       insert into treatment
 --         (treatment_id,
 --          patient_id,
 --          initial_visit_date,
 --          last_updated,
 --          last_updated_by)
 --       values
 --         (1, v_vPatientID, sysdate, sysdate, pi_nUserID);
 --       commit;

        --add all the events
        /*pck_patient_events.addallevents(pi_vSessionID,
                                        pi_vSessionClientIP,
                                        pi_nUserID,
                                        v_vPatientID,
                                        po_nStatusCode,
                                        po_vStatusComment);*/

        --insert patient step
        /*PCK_PATIENT_TX_STEP.InsertPatientStep(pi_vSessionID,
                                              pi_vSessionClientIP,
                                              pi_nUserID,
                                              v_vPatientID,
                                              0,
                                              po_nStatusCode,
                                              po_vStatusComment);*/

       --assign assessments
       /*PCK_INTAKE.AssignInitialAssessments(pi_vSessionID,
                                              pi_vSessionClientIP,
                                              pi_nUserID,
                                              v_vPatientID,
                                              po_nStatusCode,
                                              po_vStatusComment);*/
   else
                                              
       select t.patient_id into v_vPatientID 
       from patient_demographics t   
       where t.region_id = pi_nRegionID 
       and t.site_id = pi_nSiteID 
       and t.dfn = pi_vMDWSPatientID;
   
        --pass the patient id back to the caller
        po_vPatientID := v_vPatientID;    
        
        --todo: not updating the patients record at this time                                                                                
        null;

   end if;

exception
    when others then
      po_nStatusCode    := PCK_COMMON.c_nStatus_Error;
      po_vStatusComment := '';
end;
      
END;
/

